import React, { useState } from 'react';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from './ui/dialog';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { clinicInfo, services } from '../mockData';
import { Calendar, MessageSquare } from 'lucide-react';
import './BookingDialog.css';

const BookingDialog = ({ isOpen, onClose }) => {
  const [formData, setFormData] = useState({
    name: '',
    phone: '',
    email: '',
    service: '',
    preferredDate: '',
    notes: ''
  });

  const handleChange = (e) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    });
  };

  const handleServiceChange = (value) => {
    setFormData({
      ...formData,
      service: value
    });
  };

  const handleWhatsAppBooking = () => {
    const message = `Hola ReVive, me gustaría agendar una consulta:

Nombre: ${formData.name}
Teléfono: ${formData.phone}
Email: ${formData.email}
Servicio de interés: ${formData.service || 'No especificado'}
Fecha preferida: ${formData.preferredDate || 'Flexible'}
Notas: ${formData.notes || 'Ninguna'}`;

    const encodedMessage = encodeURIComponent(message);
    const whatsappUrl = `https://wa.me/${clinicInfo.whatsappNumber}?text=${encodedMessage}`;
    
    window.open(whatsappUrl, '_blank');
    onClose();
    setFormData({ name: '', phone: '', email: '', service: '', preferredDate: '', notes: '' });
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="booking-dialog">
        <DialogHeader>
          <DialogTitle className="dialog-title">
            <Calendar className="title-icon" />
            Agendar Consulta
          </DialogTitle>
          <DialogDescription className="dialog-description">
            Completa el formulario y te contactaremos por WhatsApp para confirmar tu cita
          </DialogDescription>
        </DialogHeader>
        <div className="booking-form">
          <div className="form-group">
            <Label htmlFor="booking-name">Nombre Completo *</Label>
            <Input
              id="booking-name"
              name="name"
              value={formData.name}
              onChange={handleChange}
              placeholder="Tu nombre"
              required
            />
          </div>
          <div className="form-group">
            <Label htmlFor="booking-phone">Teléfono / WhatsApp *</Label>
            <Input
              id="booking-phone"
              name="phone"
              type="tel"
              value={formData.phone}
              onChange={handleChange}
              placeholder="+52 123 456 7890"
              required
            />
          </div>
          <div className="form-group">
            <Label htmlFor="booking-email">Correo Electrónico</Label>
            <Input
              id="booking-email"
              name="email"
              type="email"
              value={formData.email}
              onChange={handleChange}
              placeholder="tu@email.com"
            />
          </div>
          <div className="form-group">
            <Label htmlFor="booking-service">Servicio de Interés</Label>
            <Select onValueChange={handleServiceChange} value={formData.service}>
              <SelectTrigger>
                <SelectValue placeholder="Selecciona un servicio" />
              </SelectTrigger>
              <SelectContent>
                {services.map((service) => (
                  <SelectItem key={service.id} value={service.title}>
                    {service.title}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div className="form-group">
            <Label htmlFor="booking-date">Fecha Preferida</Label>
            <Input
              id="booking-date"
              name="preferredDate"
              type="date"
              value={formData.preferredDate}
              onChange={handleChange}
            />
          </div>
          <div className="form-group">
            <Label htmlFor="booking-notes">Notas Adicionales</Label>
            <Textarea
              id="booking-notes"
              name="notes"
              value={formData.notes}
              onChange={handleChange}
              placeholder="Cuéntanos sobre tu situación..."
              rows={3}
            />
          </div>
          <Button
            onClick={handleWhatsAppBooking}
            className="btn-primary w-full"
            disabled={!formData.name || !formData.phone}
          >
            <MessageSquare size={18} />
            Continuar por WhatsApp
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
};

export default BookingDialog;
