import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import { Label } from '../components/ui/label';
import { Textarea } from '../components/ui/textarea';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../components/ui/tabs';
import { 
  services as initialServices,
  benefits as initialBenefits,
  testimonials as initialTestimonials,
  faqs as initialFaqs,
  clinicInfo as initialClinicInfo,
  doctorInfo as initialDoctorInfo
} from '../mockData';
import { LogOut, Save, Edit, Trash2 } from 'lucide-react';
import './AdminDashboard.css';

const AdminDashboard = () => {
  const navigate = useNavigate();
  const [activeTab, setActiveTab] = useState('clinic');
  
  // State for editable data
  const [clinicInfo, setClinicInfo] = useState(initialClinicInfo);
  const [doctorInfo, setDoctorInfo] = useState(initialDoctorInfo);
  const [services, setServices] = useState(initialServices);
  const [benefits, setBenefits] = useState(initialBenefits);
  const [testimonials, setTestimonials] = useState(initialTestimonials);
  const [faqs, setFaqs] = useState(initialFaqs);

  useEffect(() => {
    const isAuth = localStorage.getItem('adminAuth');
    if (!isAuth) {
      navigate('/admin/login');
    }
  }, [navigate]);

  const handleLogout = () => {
    localStorage.removeItem('adminAuth');
    navigate('/admin/login');
  };

  const handleSave = () => {
    // Mock save - will be connected to backend later
    console.log('Saving data...', {
      clinicInfo,
      doctorInfo,
      services,
      benefits,
      testimonials,
      faqs
    });
    alert('Cambios guardados exitosamente (Mock - se conectará al backend)');
  };

  const updateClinicInfo = (field, value) => {
    setClinicInfo({ ...clinicInfo, [field]: value });
  };

  const updateDoctorInfo = (field, value) => {
    setDoctorInfo({ ...doctorInfo, [field]: value });
  };

  const updateService = (id, field, value) => {
    setServices(services.map(s => s.id === id ? { ...s, [field]: value } : s));
  };

  const updateBenefit = (id, field, value) => {
    setBenefits(benefits.map(b => b.id === id ? { ...b, [field]: value } : b));
  };

  const updateTestimonial = (id, field, value) => {
    setTestimonials(testimonials.map(t => t.id === id ? { ...t, [field]: value } : t));
  };

  const updateFaq = (id, field, value) => {
    setFaqs(faqs.map(f => f.id === id ? { ...f, [field]: value } : f));
  };

  return (
    <div className="admin-dashboard">
      <div className="dashboard-header">
        <div className="container">
          <div className="header-content">
            <div>
              <h1 className="heading-2">Panel de Administración</h1>
              <p className="body-small">ReVive - Gestión de Contenido</p>
            </div>
            <div className="header-actions">
              <Button onClick={handleSave} className="btn-primary">
                <Save size={18} />
                Guardar Cambios
              </Button>
              <Button onClick={handleLogout} variant="outline">
                <LogOut size={18} />
                Cerrar Sesión
              </Button>
            </div>
          </div>
        </div>
      </div>

      <div className="dashboard-content">
        <div className="container">
          <Tabs value={activeTab} onValueChange={setActiveTab} className="dashboard-tabs">
            <TabsList className="tabs-list">
              <TabsTrigger value="clinic">Información Clínica</TabsTrigger>
              <TabsTrigger value="doctor">Dr. M</TabsTrigger>
              <TabsTrigger value="services">Servicios</TabsTrigger>
              <TabsTrigger value="benefits">Beneficios</TabsTrigger>
              <TabsTrigger value="testimonials">Testimonios</TabsTrigger>
              <TabsTrigger value="faqs">FAQs</TabsTrigger>
            </TabsList>

            {/* Clinic Info Tab */}
            <TabsContent value="clinic">
              <Card>
                <CardHeader>
                  <CardTitle>Información de la Clínica</CardTitle>
                  <CardDescription>Edita la información básica de contacto</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="edit-form">
                    <div className="form-group">
                      <Label>Nombre de la Clínica</Label>
                      <Input
                        value={clinicInfo.name}
                        onChange={(e) => updateClinicInfo('name', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Eslogan</Label>
                      <Input
                        value={clinicInfo.tagline}
                        onChange={(e) => updateClinicInfo('tagline', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Teléfono</Label>
                      <Input
                        value={clinicInfo.phone}
                        onChange={(e) => updateClinicInfo('phone', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Email</Label>
                      <Input
                        value={clinicInfo.email}
                        onChange={(e) => updateClinicInfo('email', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Dirección</Label>
                      <Textarea
                        value={clinicInfo.address}
                        onChange={(e) => updateClinicInfo('address', e.target.value)}
                        rows={2}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Número de WhatsApp (sin espacios)</Label>
                      <Input
                        value={clinicInfo.whatsappNumber}
                        onChange={(e) => updateClinicInfo('whatsappNumber', e.target.value)}
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Doctor Info Tab */}
            <TabsContent value="doctor">
              <Card>
                <CardHeader>
                  <CardTitle>Información del Dr. M</CardTitle>
                  <CardDescription>Edita la información del especialista</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="edit-form">
                    <div className="form-group">
                      <Label>Nombre del Doctor</Label>
                      <Input
                        value={doctorInfo.name}
                        onChange={(e) => updateDoctorInfo('name', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Título</Label>
                      <Input
                        value={doctorInfo.title}
                        onChange={(e) => updateDoctorInfo('title', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Especialización</Label>
                      <Input
                        value={doctorInfo.specialization}
                        onChange={(e) => updateDoctorInfo('specialization', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Certificaciones</Label>
                      <Input
                        value={doctorInfo.certifications}
                        onChange={(e) => updateDoctorInfo('certifications', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Experiencia</Label>
                      <Input
                        value={doctorInfo.experience}
                        onChange={(e) => updateDoctorInfo('experience', e.target.value)}
                      />
                    </div>
                    <div className="form-group">
                      <Label>Descripción</Label>
                      <Textarea
                        value={doctorInfo.description}
                        onChange={(e) => updateDoctorInfo('description', e.target.value)}
                        rows={5}
                      />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            {/* Services Tab */}
            <TabsContent value="services">
              <div className="cards-grid">
                {services.map((service) => (
                  <Card key={service.id}>
                    <CardHeader>
                      <CardTitle className="flex items-center justify-between">
                        <span>Servicio #{service.id}</span>
                        <Edit size={18} className="text-muted" />
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="edit-form">
                        <div className="form-group">
                          <Label>Título</Label>
                          <Input
                            value={service.title}
                            onChange={(e) => updateService(service.id, 'title', e.target.value)}
                          />
                        </div>
                        <div className="form-group">
                          <Label>Descripción</Label>
                          <Textarea
                            value={service.description}
                            onChange={(e) => updateService(service.id, 'description', e.target.value)}
                            rows={4}
                          />
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </TabsContent>

            {/* Benefits Tab */}
            <TabsContent value="benefits">
              <div className="cards-grid">
                {benefits.map((benefit) => (
                  <Card key={benefit.id}>
                    <CardHeader>
                      <CardTitle className="flex items-center justify-between">
                        <span>Beneficio #{benefit.id}</span>
                        <Edit size={18} className="text-muted" />
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="edit-form">
                        <div className="form-group">
                          <Label>Título</Label>
                          <Input
                            value={benefit.title}
                            onChange={(e) => updateBenefit(benefit.id, 'title', e.target.value)}
                          />
                        </div>
                        <div className="form-group">
                          <Label>Descripción</Label>
                          <Textarea
                            value={benefit.description}
                            onChange={(e) => updateBenefit(benefit.id, 'description', e.target.value)}
                            rows={3}
                          />
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </TabsContent>

            {/* Testimonials Tab */}
            <TabsContent value="testimonials">
              <div className="cards-grid">
                {testimonials.map((testimonial) => (
                  <Card key={testimonial.id}>
                    <CardHeader>
                      <CardTitle className="flex items-center justify-between">
                        <span>{testimonial.name}</span>
                        <Edit size={18} className="text-muted" />
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="edit-form">
                        <div className="form-group">
                          <Label>Nombre</Label>
                          <Input
                            value={testimonial.name}
                            onChange={(e) => updateTestimonial(testimonial.id, 'name', e.target.value)}
                          />
                        </div>
                        <div className="form-group">
                          <Label>Edad</Label>
                          <Input
                            type="number"
                            value={testimonial.age}
                            onChange={(e) => updateTestimonial(testimonial.id, 'age', parseInt(e.target.value))}
                          />
                        </div>
                        <div className="form-group">
                          <Label>Condición</Label>
                          <Input
                            value={testimonial.condition}
                            onChange={(e) => updateTestimonial(testimonial.id, 'condition', e.target.value)}
                          />
                        </div>
                        <div className="form-group">
                          <Label>Testimonio</Label>
                          <Textarea
                            value={testimonial.text}
                            onChange={(e) => updateTestimonial(testimonial.id, 'text', e.target.value)}
                            rows={4}
                          />
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </TabsContent>

            {/* FAQs Tab */}
            <TabsContent value="faqs">
              <div className="cards-grid">
                {faqs.map((faq) => (
                  <Card key={faq.id}>
                    <CardHeader>
                      <CardTitle className="flex items-center justify-between">
                        <span>FAQ #{faq.id}</span>
                        <Edit size={18} className="text-muted" />
                      </CardTitle>
                    </CardHeader>
                    <CardContent>
                      <div className="edit-form">
                        <div className="form-group">
                          <Label>Pregunta</Label>
                          <Input
                            value={faq.question}
                            onChange={(e) => updateFaq(faq.id, 'question', e.target.value)}
                          />
                        </div>
                        <div className="form-group">
                          <Label>Respuesta</Label>
                          <Textarea
                            value={faq.answer}
                            onChange={(e) => updateFaq(faq.id, 'answer', e.target.value)}
                            rows={5}
                          />
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </div>
  );
};

export default AdminDashboard;
